unit ButtonPanel;

{-----------------------------------------------------------------------}
{ Programmer    Kevin S. Gallagher   gallaghe@teleport.com              }
{ Version       1.0.1                                                   }
{ Description   This is a TPanel which allows you to align all or some  }
{               TButtons left/center/right.                             }
{                                                                       }
{ Caveat        Alignment works at design time.                         }
{                                                                       }
{ Tested with   D3, D4 (see below about D5)                             }
{                                                                       }
{ Copyright/Usage etc.                                                  }
{ use at Your own risk                                                  }
{ NOT TO BE SOLD OR DISTRIBUTED FOR A CHARGE without my permission.     }
{                                                                       }
{ How to use                                                            }
{ 1. Drop the panel on a form                                           }
{ 2. Add one or more TButtons to the panel                              }
{ 3. Add names of buttons to <Buttons>, first in last out               }
{ 4. Set alignment for buttons <ButtonAlignment>                        }
{    Using:                                                             }
{    a) The Object Inspector                                            }
{    b) Right click the component at design time.                       }
{    c) By coding it.                                                   }
{                                                                       }
{ 5. Run the project                                                    }
{                                                                       }
{                                                                       }
{ NOTE:                                                                 }
{ I orginally coded this for D3/D4 and will work with D5. I suggest if  }
{ using D5 make the following change before installing. Change          }
{ $IFDEF VER120 to $IFDEF VER130                                        }
{-----------------------------------------------------------------------}
{ Revision history                                                      }
{ V1.0.0    05/14/1999  Intial release                                  }
{ V1.0.1    05/22/1999  Moved all properties beneath ButtonOptions      }
{                       Reversed For loop for baLeft alignment          }
{                       Added MenuItem to IDE context menu              }
{                       Added vertical aligning of buttons to the panel }
{           07/23/2000  Changed some documentation.                     }
{-----------------------------------------------------------------------}






interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, Dialogs, stdctrls,
  ComCtrls, DsgnIntf,
  Extctrls;

type
  BAlignment = ( baLeft, baRight, baCenter ) ;

  TAlignOptions = ( doAlignTop, doAlignLeft ) ;

  TOptions = set of TAlignOptions ;

  TAbout = class(TPropertyEditor)
  public
    procedure Edit; override ;
    function GetAttributes: TPropertyAttributes; override ;
    function GetValue: string; override ;
  end;

  TKG_ButtonPanel = class ;

  TButtonOptions = class(TPersistent)
  private
    FAlignButtons  :Boolean ;
    FButtonSpacing :Integer ;
    FButtonHeight  :Integer ;
    FButtonWidth   :Integer ;
    FbAlignment    :BAlignment ;
    FVerticalAlign :Boolean ;
    FTop           :Integer ;
    FOptions       :TOptions ;
    procedure SetButtonSpacing(Value:Integer) ;
    procedure SetOptions(Value: TOptions) ;
  public
    constructor Create ;
  published
    property AlignButtons   :Boolean         read FAlignButtons  write FAlignButtons     default True ;
    property ButtonAlignment:BAlignment      read FbAlignment    write FbAlignment       default baRight ;
    property ButtonSpacing  :Integer         read FButtonSpacing write SetButtonSpacing ;
    property ButtonHeight   :Integer         read FButtonHeight  write FButtonHeight ;
    property ButtonWidth    :Integer         read FButtonWidth   write FButtonWidth ;
    property Top            :Integer         read FTop           write FTop ;
    property VerticalAlign  :Boolean         read FVerticalAlign write FVerticalAlign    default False ;
    property Options: TOptions read FOptions write SetOptions                            default [] ;
  end ;

  TKG_ButtonPanel = class(TPanel)
  private
    FAbout         :TAbout ;
    FButtonList    :TStringList ;
    FButtonOptions :TButtonOptions ;
    procedure SetButtonOptions(Value: TButtonOptions) ;
    procedure ButtonList(S:TStringList) ;
  public
    constructor Create(AOwner: TComponent) ; override ;
    destructor Destroy ; override ;
    procedure Resize ; override ;
  published
     property Align                                                                       default alBottom ;
     property Alignment                                                                   default taLeftJustify ;
     property BevelOuter                                                                  default bvNone ;
     property About          :TAbout          read FAbout         write FAbout ;     
     property Buttons        :TStringList     read FButtonList    write ButtonList ;
     property ButtonOptions  :TButtonOptions  read FButtonOptions write SetButtonOptions ;
  end;

  TKG_Selector = class(TComponentEditor)
  public
    function GetVerbCount: Integer; override ;
    function GetVerb(Index: Integer): string; override ;
    procedure ExecuteVerb(Index: Integer); override ;
  end ;

procedure Register ;

var
  FParent:TKG_ButtonPanel ;

implementation

{$R *.RES}

procedure TAbout.Edit ;
begin
  Application.MessageBox('ButtonAlignPanel component v1.0.1'#13'by Kevin S. Gallagher',
  'About ButtonAlignPanel Component', MB_OK + MB_ICONINFORMATION) ;
end ;

function TAbout.GetAttributes: TPropertyAttributes ;
begin
  Result:= [paMultiSelect, paDialog, paReadOnly] ;
end ;

function TAbout.GetValue: string ;
begin
  Result:= '(about)' ;
end ;

constructor TButtonOptions.Create;
begin
  inherited Create ;
  ButtonSpacing := 5 ;
  ButtonWidth := 75 ;
  ButtonHeight := 25 ;
  AlignButtons := True ;
end;

procedure TButtonOptions.SetButtonSpacing(Value :Integer) ;
begin
  if Value <> FButtonSpacing then begin
    FButtonSpacing := Value ;          { KSG Trial }
  end ;
end ;

procedure TButtonOptions.SetOptions(Value: TOptions) ;
begin
  if Value <> fOptions then begin
    if (doAlignTop in Value) and not (doAlignTop in FOptions) then
      Value := Value + [doAlignTop]
    else if not (doAlignTop in Value) and (doAlignTop in FOptions) then
      Value := Value - [doAlignTop] ;

    if (doAlignLeft in Value) and not (doAlignLeft in FOptions) then
      Value := Value + [doAlignLeft]
    else if not (doAlignLeft in Value) and (doAlignLeft in FOptions) then
      Value := Value - [doAlignLeft] ;

    FOptions := Value ;
  end ;
end ;

constructor TKG_ButtonPanel.Create(AOwner: TComponent) ;
begin
  inherited Create( AOwner ) ;

  FButtonList := TStringList.Create ;
  ControlStyle := ControlStyle - [csSetCaption] ;
  Align := alBottom ;
  Alignment := taLeftJustify ;
  BevelOuter := bvNone ;

  FButtonOptions  := TButtonOptions.Create ;
  FButtonOptions.FButtonSpacing := 5 ;
  FButtonOptions.FButtonWidth := 75 ;
  FButtonOptions.FButtonHeight := 25 ;
  FButtonOptions.FbAlignment := baRight ;
  FButtonOptions.AlignButtons := True ;
  FButtonOptions.FTop := 0 ;

  Height := 25 ;
end ;

destructor TKG_ButtonPanel.Destroy ;
begin
  inherited Destroy ;
  FButtonList.Free ;
end ;

procedure TKG_ButtonPanel.SetButtonOptions(Value: TButtonOptions) ;
begin
  { Forces properties to be written to DFM file }
end ;

procedure TKG_ButtonPanel.ButtonList(S: TStringList) ;
begin
  FButtonList.Assign(S) ;
  FButtonList.Text := UpperCase(FButtonList.Text) ;
end ;

procedure TKG_ButtonPanel.Resize ;
var
  {$IFDEF VER120}
  B:TWinControl ;
  {$ENDIF}
  S:String ;
  x:Integer ;
  y:Integer ;
  i:Integer ;
  t:Integer ;
  iSpacing:Integer ;
  iLeft:Integer ;
  iTop:Integer ;
  ButtonWidth,
  ButtonHeight,
  TotalButtonHeight,
  ButtonCount:Integer ;
  bFirstTime: boolean ;
begin
  inherited ;

  if ButtonOptions.AlignButtons then begin
    ButtonCount  := 0 ;
    for x := 0 to ControlCount -1 do
      if Controls[x] is TWinControl then  Inc(ButtonCount) ;

    ButtonWidth  := ButtonOptions.ButtonWidth ;
    ButtonHeight := ButtonOptions.ButtonHeight ;
    iSpacing     := ButtonOptions.ButtonSpacing ;
    iTop         := ButtonOptions.Top ;
    iLeft        := 0 ;
    bFirstTime   := True ;

    if ButtonOptions.VerticalAlign then begin
      TotalButtonHeight := (ButtonCount * ButtonHeight) + ((ButtonCount -1) * iSpacing) ;
      if  TotalButtonHeight > ClientHeight then
        exit ;

      if doAlignTop in ButtonOptions.Options then
        iTop := (ClientHeight - TotalButtonHeight) div 2 ;

      if doAlignLeft in ButtonOptions.Options then
        iLeft := (ClientWidth - ButtonWidth) div 2 ;

      {$IFDEF VER120}
      for x := 0 to FButtonList.Count -1 do begin
        B := TWinControl(FindChildControl(FButtonList.Strings[x])) ;
        if B <> nil then begin
          if bFirstTime then begin
            if not (doAlignLeft in ButtonOptions.Options) then
              iLeft := B.Left ;
            bFirstTime := False ;
          end else
            iTop := iTop + iSpacing + ButtonHeight ;

          if doAlignLeft in ButtonOptions.Options then
            B.SetBounds(iLeft,iTop,ButtonWidth,ButtonHeight)
          else
            B.SetBounds(B.Left,iTop,ButtonWidth,ButtonHeight) ;
        end ;
      end ;
      {$ELSE}
      for x := FButtonList.Count -1 downto 0  do begin
        S := FButtonList.Strings[x] ;
        for i := 0 to ControlCount - 1 do
          if UpperCase(Controls[i].Name) = S then begin
            if Controls[i] is TWinControl then begin
              (Controls[i] as TWinControl).Top := iTop ;
              (Controls[i] as TWinControl).Width := ButtonWidth ;
              (Controls[i] as TWinControl).Height := ButtonHeight ;
              (Controls[i] as TWinControl).Left := iLeft ;
              iTop := iTop + iSpacing + ButtonHeight ;
            end ;
          end ;
      end ;
      {$ENDIF}
    end else begin
      if doAlignTop in ButtonOptions.Options then
        iTop := (ClientHeight - ButtonHeight) div 2 ;

      case ButtonOptions.FbAlignment of
        baRight:
        begin
          iLeft := Width - iSpacing ;
          t := iLeft ;
          {$IFDEF VER120}
          for x := 0 to FButtonList.Count -1 do begin
            B := TWinControl(FindChildControl(FButtonList.Strings[x])) ;
            if B <> nil then begin
              if iLeft = t then   iLeft := iLeft - (B.Width) ;

              B.SetBounds(iLeft,iTop,ButtonWidth,ButtonHeight) ;
              iLeft := iLeft - (B.Width) - iSpacing ;
            end ;
          end ;
          {$ELSE}
          for x := 0 to FButtonList.Count -1 do begin
            S := FButtonList.Strings[x] ;
            for i := 0 to ControlCount - 1 do
              if UpperCase(Controls[i].Name) = S then begin
                if Controls[i] is TWinControl then begin
                  (Controls[i] as TWinControl).Top := iTop ;
                  (Controls[i] as TWinControl).Width := ButtonWidth ;
                  (Controls[i] as TWinControl).Height := ButtonHeight ;
                  if iLeft = t then   iLeft := iLeft - ((Controls[i] as TWinControl).Width) ;
                  (Controls[i] as TWinControl).Left := iLeft ;
                  iLeft := iLeft - ((Controls[i] as TWinControl).Width) - iSpacing ;
                end ;
              end ;
          end ;
          {$ENDIF}
        end ;
        baLeft:
        begin
          iLeft := iSpacing ;
          {$IFDEF VER120}
          for x := FButtonList.Count -1 downto 0  do begin
            B := TWinControl(FindChildControl(FButtonList.Strings[x])) ;
            if B <> nil then begin
              B.SetBounds(iLeft,iTop,ButtonWidth,ButtonHeight) ;
              iLeft := iLeft + (B.Width) + iSpacing ;
            end ;
          end ;
          {$ELSE}
          for x := FButtonList.Count -1 downto 0  do begin
            S := FButtonList.Strings[x] ;
            for i := 0 to ControlCount - 1 do
              if UpperCase(Controls[i].Name) = S then begin
                if Controls[i] is TWinControl then begin
                  (Controls[i] as TWinControl).Top := iTop ;
                  (Controls[i] as TWinControl).Width := ButtonWidth ;
                  (Controls[i] as TWinControl).Height := ButtonHeight ;
                  (Controls[i] as TWinControl).Left := iLeft ;
                  iLeft := iLeft + ((Controls[i] as TWinControl).Width) + iSpacing ;
                end ;
              end ;
          end ;
          {$ENDIF}
        end ;
        baCenter:
        begin
          if ButtonCount = 0 then exit ;

          Y := (ClientWidth - (ButtonWidth * ButtonCount + iSpacing * ButtonCount)) div 2 ;
          for x := FButtonList.Count -1 downto 0  do begin
            S := FButtonList.Strings[x] ;
            for i := 0 to ControlCount - 1 do
              if UpperCase(Controls[i].Name) = S then begin
                if Controls[i] is TWinControl then begin
                  Controls[i].SetBounds(Y,iTop,ButtonWidth, ButtonHeight) ;
                  Inc(Y, ButtonWidth + iSpacing) ;
                end ;
              end ;
          end ;
        end;
      end ;
    end ;
  end ;
end ;

function TKG_Selector.GetVerbCount: Integer ;
begin
  Result := 1 ;
end ;

function TKG_Selector.GetVerb(Index:Integer): string ;
begin
  case index of
    0: Result := 'Align Buttons' ;
  end ;
end ;

procedure TKG_Selector.ExecuteVerb(Index:Integer) ;
begin
  case Index of
    0: TKG_ButtonPanel(Component).Resize ; // Perform(WM_SIZE,0,0) ;
  end ;
  Designer.Modified ;
end ;



procedure Register;
begin
  RegisterComponents('Samples', [TKG_ButtonPanel]) ;
  RegisterPropertyEditor(TypeInfo(TAbout), TKG_ButtonPanel, 'ABOUT', TAbout) ;
  RegisterComponentEditor(TKG_ButtonPanel,TKG_Selector) ;
end ;

end.
